import serial
from CubeRawCommands import *

# Constants
CUBE_LAYER_COUNT = 8
CUBE_ROW_COUNT = 8
CUBE_COLUMN_COUNT = 8
CUBE_ROTATIONS = 7
GCS_REG_SIZE = 36
GCS_LAYER_SIZE = GCS_REG_SIZE * CUBE_ROW_COUNT

# 3 bytes per LED, 1536 total for the cube
cube_buffer = bytearray(CUBE_LAYER_COUNT*GCS_LAYER_SIZE)
rotation_counter = 0

# Specify the serial port to connect to
serial_port = serial.Serial()

'''Opens the serial port for sending commands through.'''
def Cube_Init(port, baudrate):
    serial_port.port = port
    serial_port.baudrate = baudrate
    serial_port.open()

'''Clears the local buffer for LED values.'''
def Cube_Clear():
    for i in range(len(cube_buffer)):
        cube_buffer[i] = 0

'''Sets a specific pixel in the local buffer.'''
def Cube_Set_Pixel(layer, row, column, R, G, B):
    # Set the specified pixel to the given color
    R &= 0x0FFF
    G &= 0x0FFF
    B &= 0x0FFF
    var = row * GCS_REG_SIZE + (column // 2 * 9)
    offset = (layer * GCS_LAYER_SIZE) + var
    if column % 2 == 0:
        cube_buffer[offset+0] = R & 0xFF
        cube_buffer[offset+1] = ((G << 4) | (R >> 8)) & 0xFF
        cube_buffer[offset+2] = G >> 4
        cube_buffer[offset+3] = B & 0xFF
        cube_buffer[offset+4] = ((cube_buffer[offset+4] & 0xF0) | (B >> 8)) & 0xFF
    elif column % 2 == 1:
        cube_buffer[offset+4] = ((cube_buffer[offset+4] & 0x0F) | (R << 4)) & 0xFF
        cube_buffer[offset+5] = R >> 4
        cube_buffer[offset+6] = G & 0xFF
        cube_buffer[offset+7] = ((B << 4) | (G >> 8)) & 0xFF
        cube_buffer[offset+8] = B >> 4

'''Retreives a specific pixel in the local buffer.'''
def Cube_Get_Pixel(layer, row, column):
    # Get and return the color for the specified pixel
    R = G = B = 0
    var = row * GCS_REG_SIZE + (column // 2 * 9)
    offset = (layer * GCS_LAYER_SIZE) + var
    if column % 2 == 0:
        R = cube_buffer[offset+0] | ((cube_buffer[offset+1] & 0x0F) << 8);
        G = (cube_buffer[offset+1] >> 4) | (cube_buffer[offset+2] << 4);
        B = cube_buffer[offset+3] | ((cube_buffer[offset+4] & 0x0F) << 8);
    elif column % 2 == 1:
        R = (cube_buffer[offset+4] >> 4) | (cube_buffer[offset+5] << 4);
        G = cube_buffer[offset+6] | ((cube_buffer[offset+7] & 0x0F) << 8);
        B = (cube_buffer[offset+7] >> 4) | (cube_buffer[offset+8] << 4);
    return [R,G,B]

'''Moves a value from one pixel to another in the local buffer.'''
def Cube_Move_Pixel(layer_1, row_1, column_1, layer_2, row_2, column_2):
    old = Cube_Get_Pixel(layer1, row_1, column_1)
    Cube_Set_Pixel(layer_2, row_2, column_2, old[0], old[1], old[2])

'''Rotates the specified shell in the local buffer.'''
def Cube_Rotate_Shell(shell, direction):
    for layer in range(CUBE_LAYER_COUNT):
        if direction == 1:
            if shell == 0:
                # Rotate outermost layer
                old = Cube_Get_Pixel(layer, 0, 0);
                Cube_Move_Pixel(layer, 0, 1, layer, 0, 0);
                Cube_Move_Pixel(layer, 0, 2, layer, 0, 1);
                Cube_Move_Pixel(layer, 0, 3, layer, 0, 2);
                Cube_Move_Pixel(layer, 0, 4, layer, 0, 3);
                Cube_Move_Pixel(layer, 0, 5, layer, 0, 4);
                Cube_Move_Pixel(layer, 0, 6, layer, 0, 5);
                Cube_Move_Pixel(layer, 0, 7, layer, 0, 6);
                Cube_Move_Pixel(layer, 1, 7, layer, 0, 7);
                Cube_Move_Pixel(layer, 2, 7, layer, 1, 7);
                Cube_Move_Pixel(layer, 3, 7, layer, 2, 7);
                Cube_Move_Pixel(layer, 4, 7, layer, 3, 7);
                Cube_Move_Pixel(layer, 5, 7, layer, 4, 7);
                Cube_Move_Pixel(layer, 6, 7, layer, 5, 7);
                Cube_Move_Pixel(layer, 7, 7, layer, 6, 7);
                Cube_Move_Pixel(layer, 7, 6, layer, 7, 7);
                Cube_Move_Pixel(layer, 7, 5, layer, 7, 6);
                Cube_Move_Pixel(layer, 7, 4, layer, 7, 5);
                Cube_Move_Pixel(layer, 7, 3, layer, 7, 4);
                Cube_Move_Pixel(layer, 7, 2, layer, 7, 3);
                Cube_Move_Pixel(layer, 7, 1, layer, 7, 2);
                Cube_Move_Pixel(layer, 7, 0, layer, 7, 1);
                Cube_Move_Pixel(layer, 6, 0, layer, 7, 0);
                Cube_Move_Pixel(layer, 5, 0, layer, 6, 0);
                Cube_Move_Pixel(layer, 4, 0, layer, 5, 0);
                Cube_Move_Pixel(layer, 3, 0, layer, 4, 0);
                Cube_Move_Pixel(layer, 2, 0, layer, 3, 0);
                Cube_Move_Pixel(layer, 1, 0, layer, 2, 0);
                Cube_Set_Pixel(layer, 1, 0, old[0], old[1], old[2]);
            elif shell == 1:
                # Rotate second to outermost layer
                old = Cube_Get_Pixel(layer, 1, 1);
                Cube_Move_Pixel(layer, 1, 2, layer, 1, 1);
                Cube_Move_Pixel(layer, 1, 3, layer, 1, 2);
                Cube_Move_Pixel(layer, 1, 4, layer, 1, 3);
                Cube_Move_Pixel(layer, 1, 5, layer, 1, 4);
                Cube_Move_Pixel(layer, 1, 6, layer, 1, 5);
                Cube_Move_Pixel(layer, 2, 6, layer, 1, 6);
                Cube_Move_Pixel(layer, 3, 6, layer, 2, 6);
                Cube_Move_Pixel(layer, 4, 6, layer, 3, 6);
                Cube_Move_Pixel(layer, 5, 6, layer, 4, 6);
                Cube_Move_Pixel(layer, 6, 6, layer, 5, 6);
                Cube_Move_Pixel(layer, 6, 5, layer, 6, 6);
                Cube_Move_Pixel(layer, 6, 4, layer, 6, 5);
                Cube_Move_Pixel(layer, 6, 3, layer, 6, 4);
                Cube_Move_Pixel(layer, 6, 2, layer, 6, 3);
                Cube_Move_Pixel(layer, 6, 1, layer, 6, 2);
                Cube_Move_Pixel(layer, 5, 1, layer, 6, 1);
                Cube_Move_Pixel(layer, 4, 1, layer, 5, 1);
                Cube_Move_Pixel(layer, 3, 1, layer, 4, 1);
                Cube_Move_Pixel(layer, 2, 1, layer, 3, 1);
                Cube_Set_Pixel(layer, 2, 1, old[0], old[1], old[2]);
            elif shell == 2:
                # Rotate second to innermost layer
                old = Cube_Get_Pixel(layer, 2, 2);
                Cube_Move_Pixel(layer, 2, 3, layer, 2, 2);
                Cube_Move_Pixel(layer, 2, 4, layer, 2, 3);
                Cube_Move_Pixel(layer, 2, 5, layer, 2, 4);
                Cube_Move_Pixel(layer, 3, 5, layer, 2, 5);
                Cube_Move_Pixel(layer, 4, 5, layer, 3, 5);
                Cube_Move_Pixel(layer, 5, 5, layer, 4, 5);
                Cube_Move_Pixel(layer, 5, 4, layer, 5, 5);
                Cube_Move_Pixel(layer, 5, 3, layer, 5, 4);
                Cube_Move_Pixel(layer, 5, 2, layer, 5, 3);
                Cube_Move_Pixel(layer, 4, 2, layer, 5, 2);
                Cube_Move_Pixel(layer, 3, 2, layer, 4, 2);
                Cube_Set_Pixel(layer, 3, 2, old[0], old[1], old[2]);
            elif shell == 3:
                # Rotate innermost layer
                old = Cube_Get_Pixel(layer, 3, 3);
                Cube_Move_Pixel(layer, 3, 4, layer, 3, 3);
                Cube_Move_Pixel(layer, 4, 4, layer, 3, 4);
                Cube_Move_Pixel(layer, 4, 3, layer, 4, 4);
                Cube_Set_Pixel(layer, 4, 3, old[0], old[1], old[2]);
        else:
            if shell == 0:
                # Rotate outermost layer
                old = Cube_Get_Pixel(layer, 0, 0);
                Cube_Move_Pixel(layer, 1, 0, layer, 0, 0);
                Cube_Move_Pixel(layer, 2, 0, layer, 1, 0);
                Cube_Move_Pixel(layer, 3, 0, layer, 2, 0);
                Cube_Move_Pixel(layer, 4, 0, layer, 3, 0);
                Cube_Move_Pixel(layer, 5, 0, layer, 4, 0);
                Cube_Move_Pixel(layer, 6, 0, layer, 5, 0);
                Cube_Move_Pixel(layer, 7, 0, layer, 6, 0);
                Cube_Move_Pixel(layer, 7, 1, layer, 7, 0);
                Cube_Move_Pixel(layer, 7, 2, layer, 7, 1);
                Cube_Move_Pixel(layer, 7, 3, layer, 7, 2);
                Cube_Move_Pixel(layer, 7, 4, layer, 7, 3);
                Cube_Move_Pixel(layer, 7, 5, layer, 7, 4);
                Cube_Move_Pixel(layer, 7, 6, layer, 7, 5);
                Cube_Move_Pixel(layer, 7, 7, layer, 7, 6);
                Cube_Move_Pixel(layer, 6, 7, layer, 7, 7);
                Cube_Move_Pixel(layer, 5, 7, layer, 6, 7);
                Cube_Move_Pixel(layer, 4, 7, layer, 5, 7);
                Cube_Move_Pixel(layer, 3, 7, layer, 4, 7);
                Cube_Move_Pixel(layer, 2, 7, layer, 3, 7);
                Cube_Move_Pixel(layer, 1, 7, layer, 2, 7);
                Cube_Move_Pixel(layer, 0, 7, layer, 1, 7);
                Cube_Move_Pixel(layer, 0, 6, layer, 0, 7);
                Cube_Move_Pixel(layer, 0, 5, layer, 0, 6);
                Cube_Move_Pixel(layer, 0, 4, layer, 0, 5);
                Cube_Move_Pixel(layer, 0, 3, layer, 0, 4);
                Cube_Move_Pixel(layer, 0, 2, layer, 0, 3);
                Cube_Move_Pixel(layer, 0, 1, layer, 0, 2);
                Cube_Set_Pixel(layer, 0, 1, old[0], old[1], old[2]);
            if shell == 1:
                # Rotate second to outermost layer
                old = Cube_Get_Pixel(layer, 1, 1);
                Cube_Move_Pixel(layer, 2, 1, layer, 1, 1);
                Cube_Move_Pixel(layer, 3, 1, layer, 2, 1);
                Cube_Move_Pixel(layer, 4, 1, layer, 3, 1);
                Cube_Move_Pixel(layer, 5, 1, layer, 4, 1);
                Cube_Move_Pixel(layer, 6, 1, layer, 5, 1);
                Cube_Move_Pixel(layer, 6, 2, layer, 6, 1);
                Cube_Move_Pixel(layer, 6, 3, layer, 6, 2);
                Cube_Move_Pixel(layer, 6, 4, layer, 6, 3);
                Cube_Move_Pixel(layer, 6, 5, layer, 6, 4);
                Cube_Move_Pixel(layer, 6, 6, layer, 6, 5);
                Cube_Move_Pixel(layer, 5, 6, layer, 6, 6);
                Cube_Move_Pixel(layer, 4, 6, layer, 5, 6);
                Cube_Move_Pixel(layer, 3, 6, layer, 4, 6);
                Cube_Move_Pixel(layer, 2, 6, layer, 3, 6);
                Cube_Move_Pixel(layer, 1, 6, layer, 2, 6);
                Cube_Move_Pixel(layer, 1, 5, layer, 1, 6);
                Cube_Move_Pixel(layer, 1, 4, layer, 1, 5);
                Cube_Move_Pixel(layer, 1, 3, layer, 1, 4);
                Cube_Move_Pixel(layer, 1, 2, layer, 1, 3);
                Cube_Set_Pixel(layer, 1, 2, old[0], old[1], old[2]);
            if shell == 2:
                # Rotate second to innermost layer
                old = Cube_Get_Pixel(layer, 2, 2);
                Cube_Move_Pixel(layer, 3, 2, layer, 2, 2);
                Cube_Move_Pixel(layer, 4, 2, layer, 3, 2);
                Cube_Move_Pixel(layer, 5, 2, layer, 4, 2);
                Cube_Move_Pixel(layer, 5, 3, layer, 5, 2);
                Cube_Move_Pixel(layer, 5, 4, layer, 5, 3);
                Cube_Move_Pixel(layer, 5, 5, layer, 5, 4);
                Cube_Move_Pixel(layer, 4, 5, layer, 5, 5);
                Cube_Move_Pixel(layer, 3, 5, layer, 4, 5);
                Cube_Move_Pixel(layer, 2, 5, layer, 3, 5);
                Cube_Move_Pixel(layer, 2, 4, layer, 2, 5);
                Cube_Move_Pixel(layer, 2, 3, layer, 2, 4);
                Cube_Set_Pixel(layer, 2, 3, old[0], old[1], old[2]);
            if shell == 3:
                # Rotate innermost layer
                old = Cube_Get_Pixel(layer, 3, 3);
                Cube_Move_Pixel(layer, 4, 3, layer, 3, 3);
                Cube_Move_Pixel(layer, 4, 4, layer, 4, 3);
                Cube_Move_Pixel(layer, 3, 4, layer, 4, 4);
                Cube_Set_Pixel(layer, 3, 4, old[0], old[1], old[2]);

'''Rotates the entire cube in the local buffer.'''
def Cube_Rotate(direction):
    # Rotate outermost layer
    Cube_Rotate_Shell(0, direction);
    # Rotate second to outermost layer
    if ((rotation_counter != 1) and (rotation_counter != 5)):
        Cube_Rotate_Shell(1, direction);
    # Rotate second to innermost layer
    if ((rotation_counter != 0) and (rotation_counter != 2) and (rotation_counter != 4) and (rotation_counter != 6)):
        Cube_Rotate_Shell(2, direction);
    # Rotate innermost layer
    if ((rotation_counter == 3) or (rotation_counter == 7)):
        Cube_Rotate_Shell(3, direction);

    if (direction == 0):
        if rotation_counter == CUBE_ROTATIONS - 1:
            rotation_counter = 0
        else:
            rotation_counter = rotation_counter + 1
    else:
        if rotation_counter == 0:
            rotation_counter = CUBE_ROTATIONS - 1
        else:
            rotation_counter = rotation_counter - 1

'''Write the local buffer to the cube.'''
def Cube_Update():
    serial_port.write(CMD_Set_All(cube_buffer))